// Content script: runs on Google Maps pages
// Scrapes visible result cards and streams items to the extension

let MLS_isRunning = false;
let MLS_seen = new Set();
let MLS_location = '';

async function isAuthorized(){
  try {
    const res = await chrome.runtime.sendMessage({ type: 'mls_is_authorized' });
    return !!(res && res.ok && res.authorized);
  } catch { return false; }
}

function cleanPhone(v) {
  try {
    let s = String(v || '');
    s = s.replace(/%20/gi, '');
    s = s.replace(/\D+/g, '');
    return s;
  } catch (_) { return v; }
}

function cleanEmail(v) {
  try {
    let s = String(v || '').trim();
    s = s.replace(/%20/gi, '');
    s = s.split(/[?#]/)[0];
    s = s.replace(/(\.(png|jpg|jpeg|gif|bmp|webp|tiff|svg|ico|heic|heif))+$/i, '');
    return s;
  } catch (_) { return v; }
}

function isValidEmail(v) {
  try { return /^[A-Za-z0-9._%+-]+@[A-Za-z0-9.-]+\.[A-Za-z]{2,}$/.test(String(v||'').trim()); } catch(_) { return false; }
}

function normalizeUrl(url){
  try {
    if (!url) return null;
    let u = url;
    if (!/^https?:\/\//i.test(u)) u = `http://${u}`;
    const parsed = new URL(u);
    parsed.hash = '';
    return parsed.toString();
  } catch { return null; }
}

async function fetchEmailFromSingle(url, signal){
  try {
    const res = await chrome.runtime.sendMessage({ type: 'mls_fetch_html', url });
    if (!res || !res.ok) return null;
    const text = res.text;
    const emails = text.match(/[\w.%+-]+@[\w.-]+\.[A-Za-z]{2,}/g) || [];
    if (emails.length) {
      const pri = ['gmail.com','outlook.com','yahoo.com','hotmail.com'];
      for (const e of emails) if (pri.some(d => e.toLowerCase().endsWith(d))) return cleanEmail(e);
      return cleanEmail(emails[0]);
    }
  } catch {}
  return null;
}

async function fetchEmailFromSite(url, signal) {
  const u0 = normalizeUrl(url);
  if (!u0) return 'Not available';
  let base;
  try { base = new URL(u0).origin; } catch { base = null; }
  const candidates = [u0];
  if (base) {
    const paths = ['', '/contact', '/contacts', '/contact-us', '/contactez-nous', '/nous-contacter', '/about', '/about-us', '/a-propos'];
    for (const p of paths) candidates.push(base + p);
  }
  for (const u of candidates){
    const email = await fetchEmailFromSingle(u, signal);
    if (email) return email;
    await new Promise(r => setTimeout(r, 40));
  }
  return 'Not available';
}

function findCards() {
  // Primary card selector used in the Python script
  let cards = Array.from(document.querySelectorAll('div.Nv2PK'));
  if (cards.length === 0) {
    // Fallback: sometimes cards live under role=listitem
    cards = Array.from(document.querySelectorAll('[role="article"], [role="listitem"]')).filter(c => c.querySelector('div.NrDZNb'));
  }
  return cards;
}

function extractFromCard(card) {
  const nameEl = card.querySelector('div.NrDZNb') || card.querySelector('div.qBF1Pd') || card.querySelector('[aria-level="3"]');
  const phoneEl = card.querySelector('span.UsdlK');
  const linkEl = card.querySelector('div.THOPZb a.lcr4fd');
  const name = nameEl?.textContent?.trim() || 'Not available';
  const phone = cleanPhone(phoneEl?.textContent || 'Not available');
  const website = linkEl?.getAttribute('href') || 'Not available';
  return { name, phone, website };
}

function findSidePanelWebsiteHref(){
  const anchors = Array.from(document.querySelectorAll('a[href^="http"]'));
  // prefer anchors labeled as Website in multiple languages
  const labels = ['website','site','الموقع'];
  const byLabel = anchors.find(a => {
    const al = (a.getAttribute('aria-label') || '').toLowerCase();
    return labels.some(l => al.includes(l));
  });
  if (byLabel) return byLabel.href;
  // fallback: any external link in the right panel actions section
  const rightPanel = document.querySelector('div[role="main"], div[aria-label]');
  if (rightPanel){
    const a2 = rightPanel.querySelector('a[href^="http"]');
    if (a2) return a2.href;
  }
  return null;
}

async function openCardAndResolveWebsite(card){
  try { card.scrollIntoView({ behavior: 'instant', block: 'center' }); } catch {}
  try { card.click(); } catch {}
  // wait for side panel to update
  await new Promise(r => setTimeout(r, 350));
  const href = findSidePanelWebsiteHref();
  return href || 'Not available';
}

async function scrapeVisible(signal) {
  const cards = findCards();
  for (let i = 0; i < cards.length && MLS_isRunning; i++) {
    let { name, phone, website } = extractFromCard(cards[i]);
    if (!website || website === 'Not available') {
      const resolved = await openCardAndResolveWebsite(cards[i]);
      if (resolved) website = resolved;
    }
    const key = `${name}\n${phone}\n${website}`;
    if (MLS_seen.has(key)) continue;
    MLS_seen.add(key);

    let email = 'Not available';
    try {
      email = await fetchEmailFromSite(website, signal);
    } catch (_) {}

    const data = { Name: name, 'Phone number': phone, Website: website, Email: cleanEmail(email), Location: MLS_location };
    chrome.runtime.sendMessage({ type: 'mls_item', data });
    await new Promise(r => setTimeout(r, 60));
  }
}

async function scrapeAllVisible(signal) {
  const cards = findCards();
  for (let i = 0; i < cards.length && MLS_isRunning; i++) {
    let { name, phone, website } = extractFromCard(cards[i]);
    if (!website || website === 'Not available') {
      const resolved = await openCardAndResolveWebsite(cards[i]);
      if (resolved) website = resolved;
    }
    const key = `${name}\n${phone}\n${website}`;
    if (MLS_seen.has(key)) continue;
    MLS_seen.add(key);

    let email = 'Not available';
    try {
      email = await fetchEmailFromSite(website, signal);
    } catch (_) {}
    chrome.runtime.sendMessage({ type: 'mls_item', data: { Name: name, 'Phone number': phone, Website: website, Email: cleanEmail(email), Location: MLS_location } });
    await new Promise(r => setTimeout(r, 60));
  }
}

function detectScrollableAncestor(el){
  let cur = el;
  while (cur && cur !== document.body) {
    try {
      const style = getComputedStyle(cur);
      const overflowY = style.overflowY || style.overflow;
      const scrollable = (overflowY === 'auto' || overflowY === 'scroll');
      if (scrollable && cur.scrollHeight > cur.clientHeight + 20) return cur;
    } catch {}
    cur = cur.parentElement;
  }
  return null;
}

function resultsScrollerEl() {
  // Prefer real scrollable ancestor of a card
  const anyCard = document.querySelector('div.Nv2PK') || document.querySelector('[role="listitem"], [role="article"]');
  if (anyCard) {
    const anc = detectScrollableAncestor(anyCard);
    if (anc) return anc;
  }
  // Known containers; fallback to window
  return document.querySelector('.m6QErb.DxyBCb.kA9KIf.dS8AEf.ecceSd')
      || document.querySelector('.m6QErb[aria-label]')
      || document.querySelector('div[role="feed"]')
      || window;
}

function waitForResultsContainer(timeoutMs = 10000) {
  const start = Date.now();
  return new Promise(resolve => {
    function check() {
      const hasList = document.querySelector('.m6QErb.DxyBCb.kA9KIf.dS8AEf.ecceSd') || document.querySelector('.m6QErb[aria-label]') || document.querySelector('div[role="feed"]');
      const hasCard = document.querySelector('div.Nv2PK');
      if (hasList || hasCard) return resolve(true);
      if (Date.now() - start > timeoutMs) return resolve(false);
      setTimeout(check, 250);
    }
    check();
  });
}

function ensureScrollBadge(){
  let badge = document.getElementById('mls-scroll-badge');
  if (!badge) {
    badge = document.createElement('div');
    badge.id = 'mls-scroll-badge';
    badge.textContent = 'Auto scrolling…';
    Object.assign(badge.style, {
      position: 'fixed',
      top: '10px',
      right: '12px',
      padding: '6px 10px',
      background: 'rgba(52,168,83,0.92)',
      color: '#fff',
      font: '600 12px/1.2 system-ui,Segoe UI,Roboto,sans-serif',
      borderRadius: '999px',
      boxShadow: '0 2px 8px rgba(0,0,0,0.2)',
      zIndex: 2147483647,
      pointerEvents: 'none'
    });
    document.body.appendChild(badge);
  }
  return badge;
}

function highlightScroller(scroller, on){
  try {
    if (scroller && scroller !== window) {
      if (on) {
        scroller.dataset.mlsHighlight = '1';
        scroller.style.outline = '2px solid rgba(52,168,83,0.9)';
        scroller.style.outlineOffset = '-2px';
        scroller.style.boxShadow = 'inset 0 0 0 2px rgba(52,168,83,0.25)';
      } else if (scroller.dataset.mlsHighlight) {
        scroller.style.outline = '';
        scroller.style.outlineOffset = '';
        scroller.style.boxShadow = '';
        delete scroller.dataset.mlsHighlight;
      }
    }
  } catch {}
}

function forceFocusList(scroller){
  try {
    if (scroller && scroller !== window) {
      scroller.tabIndex = -1;
      scroller.focus({ preventScroll: false });
    } else {
      document.body?.focus();
    }
  } catch {}
}

function nudgeScroll(scroller){
  try {
    if (scroller && scroller !== window) {
      scroller.dispatchEvent(new WheelEvent('wheel', { deltaY: 1600, bubbles: true }));
      scroller.scrollTop += 1400;
      scroller.dispatchEvent(new WheelEvent('wheel', { deltaY: -300, bubbles: true }));
      scroller.scrollTop -= 260;
      // Visible page down hint
      scroller.dispatchEvent(new KeyboardEvent('keydown', { key: 'PageDown', code: 'PageDown', bubbles: true }));
    } else {
      window.scrollBy({ top: 1400, behavior: 'smooth' });
      setTimeout(() => window.scrollBy({ top: -260, behavior: 'smooth' }), 120);
    }
  } catch {}
}

async function scrollToEnd(scroller) {
  let lastCount = 0, stagnate = 0;
  const maxScrolls = 120;
  for (let s = 0; s < maxScrolls && MLS_isRunning; s++) {
    const lastCard0 = document.querySelector('div.Nv2PK:last-of-type');
    try { lastCard0?.scrollIntoView({ behavior: 'smooth', block: 'center' }); } catch(_) {}
    try {
      if (scroller && scroller !== window) {
        scroller.scrollTop += 1400;
        scroller.dispatchEvent(new WheelEvent('wheel', { deltaY: 1500, bubbles: true }));
        scroller.dispatchEvent(new KeyboardEvent('keydown', { key: 'PageDown', code: 'PageDown', bubbles: true }));
      } else {
        window.scrollBy({ top: 1500, behavior: 'smooth' });
      }
    } catch(_) {}
    await new Promise(r => setTimeout(r, 350));
    const cards = findCards();
    if (cards.length === lastCount) stagnate++; else { stagnate = 0; lastCount = cards.length; }
    if (stagnate >= 3) break; // stable end reached
  }
}

async function scrollAndScrape() {
  const controller = new AbortController();
  const signal = controller.signal;
  const scroller = resultsScrollerEl();

  // Wait until list/cards appear
  await waitForResultsContainer(12000);
  forceFocusList(scroller);
  // Add visual indicators for scrolling
  const badge = ensureScrollBadge();
  highlightScroller(scroller, true);
  // Rapid warmup: fire wheel events for ~1.2s to ensure list activates
  const start = performance.now();
  while (performance.now() - start < 800) {
    try {
      if (scroller && scroller !== window) {
        scroller.dispatchEvent(new WheelEvent('wheel', { deltaY: 1200, bubbles: true }));
        scroller.scrollTop += 900;
        scroller.dispatchEvent(new KeyboardEvent('keydown', { key: 'PageDown', code: 'PageDown', bubbles: true }));
      } else {
        window.scrollBy({ top: 900, behavior: 'auto' });
      }
    } catch {}
    await new Promise(r => setTimeout(r, 40));
  }
  // A few strong nudges
  for (let i = 0; i < 2; i++) { nudgeScroll(scroller); await new Promise(r => setTimeout(r, 90)); }

  // Phase A: scroll to the very end first (until stable)
  await scrollToEnd(scroller);

  // Phase B: scrape all visible cards sequentially with streaming
  await scrapeAllVisible(signal);

  chrome.runtime.sendMessage({ type: 'mls_done' });
  // remove visuals
  highlightScroller(scroller, false);
  if (badge && badge.remove) badge.remove();
}

chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg?.type === 'mls_ping') {
    (async () => {
      const auth = await isAuthorized();
      sendResponse({ ok: true, ready: !!auth });
    })();
    return true;
  }
  if (msg?.type === 'mls_start') {
    (async () => {
      const auth = await isAuthorized();
      if (!auth) { sendResponse({ ok: false, error: 'unauthorized' }); return; }
      MLS_isRunning = true;
      MLS_seen.clear();
      MLS_location = msg.location || '';
      scrollAndScrape();
      sendResponse({ ok: true });
    })();
    return true;
  }
  if (msg?.type === 'mls_stop') {
    MLS_isRunning = false;
    sendResponse({ ok: true });
    return true;
  }
});
