// Background service worker for Maps Leads Scraper
// Orchestrates scraping so it continues even if the popup is closed or user switches pages.

let MLS_STATE = {
  running: false,
  tabId: null,
  queue: [], // [{qs, loc}]
  data: [],
  seen: new Set(),
};

chrome.runtime.onInstalled.addListener(() => {
  chrome.storage.local.set({ mls_settings: { useGoogleEmailSearch: false }, mls_data: [] });
});

// Cross-origin fetch proxy for content script
chrome.runtime.onMessage.addListener((msg, sender, sendResponse) => {
  if (msg?.type === 'mls_fetch_html' && typeof msg.url === 'string') {
    fetch(msg.url, { method: 'GET', cache: 'no-store' })
      .then(r => r.text().then(text => ({ ok: r.ok, status: r.status, text })))
      .then(res => sendResponse(res))
      .catch(err => sendResponse({ ok: false, error: String(err) }));
    return true; // async
  }
  if (msg?.type === 'mls_item' && msg.data) {
    // stream from content: persist and broadcast
    const key = `${(msg.data['Name']||'').trim().toLowerCase()}|${String(msg.data['Phone number']||'').trim()}|${String(msg.data['Website']||'').trim()}|${String(msg.data['Location']||'').trim().toLowerCase()}`;
    if (!MLS_STATE.seen.has(key)) {
      MLS_STATE.seen.add(key);
      MLS_STATE.data.push(msg.data);
      chrome.storage.local.set({ mls_data: MLS_STATE.data.slice(-5000) });
      chrome.runtime.sendMessage({ type: 'mls_item', data: msg.data });
    }
  }
  if (msg?.type === 'mls_done') {
    // content signals pair finished; background will move to next automatically
  }
  if (msg?.type === 'mls_begin_job') {
    const { searches = [], locations = [] } = msg;
    beginJob(sanitize(searches), sanitize(locations));
    sendResponse({ ok: true });
    return true;
  }
  if (msg?.type === 'mls_pause_job') {
    pauseJob();
    sendResponse({ ok: true });
    return true;
  }
  if (msg?.type === 'mls_resume_job') {
    resumeJob();
    sendResponse({ ok: true });
    return true;
  }
  if (msg?.type === 'mls_stop_job') {
    stopJob();
    sendResponse({ ok: true });
    return true;
  }
  if (msg?.type === 'mls_get_data') {
    sendResponse({ ok: true, data: MLS_STATE.data || [] });
    return true;
  }
  if (msg?.type === 'mls_get_status') {
    sendResponse({ ok: true, status: { running: !!MLS_STATE.running, queueLength: (MLS_STATE.queue||[]).length } });
    return true;
  }
  if (msg?.type === 'mls_is_authorized') {
    (async () => {
      try {
        const ok = await isAuthorized();
        sendResponse({ ok: true, authorized: !!ok });
      } catch {
        sendResponse({ ok: true, authorized: false });
      }
    })();
    return true;
  }
});

function sanitize(list){ return (list||[]).map(x => String(x||'').trim()).filter(Boolean); }

async function isAuthorized(){
  try {
    const st = await chrome.storage.local.get(['mls_auth']);
    const auth = st.mls_auth || {};
    if (!auth.token) return false;
    const status = String(auth.status || 'Active').toLowerCase();
    if (status !== 'active') return false;
    if (auth.expirationDate) {
      const exp = new Date(auth.expirationDate).getTime();
      if (Number.isFinite(exp) && exp < Date.now()) return false;
    }
    return true;
  } catch { return false; }
}

async function ensureMapsTab(){
  // reuse existing maps tab if active tab is maps, else create a new one
  const tabs = await chrome.tabs.query({ url: ['https://www.google.com/maps/*','https://maps.google.com/*'] });
  let t = tabs[0];
  if (!t) t = await chrome.tabs.create({ url: 'https://www.google.com/maps', active: true });
  MLS_STATE.tabId = t.id;
  await waitForTabComplete(t.id, 20000);
  return t.id;
}

async function navigateSearch(tabId, qs, loc){
  await chrome.tabs.update(tabId, { url: `https://www.google.com/maps/search/${encodeURIComponent(qs + ' ' + loc)}?hl=en` });
  await waitForTabComplete(tabId, 30000);
  const ready = await waitForContentReady(tabId, 15000);
  return !!ready;
}

function waitForTabComplete(tabId, timeoutMs = 20000){
  const start = Date.now();
  return new Promise(resolve => {
    const timer = setInterval(async () => {
      const [t] = await chrome.tabs.query({ active: true, currentWindow: true });
      if (t && t.id === tabId && t.status === 'complete') { clearInterval(timer); resolve(true); }
      else if (Date.now() - start > timeoutMs) { clearInterval(timer); resolve(false); }
    }, 200);
  });
}

async function waitForContentReady(tabId, timeoutMs = 10000){
  const start = Date.now();
  while (Date.now() - start < timeoutMs){
    try {
      const res = await chrome.tabs.sendMessage(tabId, { type: 'mls_ping' });
      if (res && res.ok && res.ready) return true;
    } catch {}
    await new Promise(r => setTimeout(r, 300));
  }
  return false;
}

async function beginJob(searches, locations){
  if (!(await isAuthorized())) { MLS_STATE.running = false; return; }
  if (!searches.length || !locations.length) return;
  MLS_STATE.running = true;
  // Rebuild queue from inputs; preserve any remaining items if already present by appending new ones
  const newQueue = [];
  for (const qs of searches) for (const loc of locations) newQueue.push({ qs, loc });
  if (!Array.isArray(MLS_STATE.queue) || !MLS_STATE.queue.length) MLS_STATE.queue = newQueue;
  else MLS_STATE.queue = MLS_STATE.queue.concat(newQueue);
  // preserve current data; do not reset on every run unless needed
  const tabId = await ensureMapsTab();
  runQueue(tabId);
}

async function runQueue(tabId){
  while (MLS_STATE.running && MLS_STATE.queue.length){
    if (!(await isAuthorized())) { MLS_STATE.running = false; break; }
    const { qs, loc } = MLS_STATE.queue.shift();
    const ok = await navigateSearch(tabId, qs, loc);
    if (!ok) continue;
    try { await chrome.tabs.sendMessage(tabId, { type: 'mls_start', location: loc }); } catch {}
    await waitForPairDone(180000); // up to 3 minutes per pair
    // small pacing
    await new Promise(r => setTimeout(r, 300));
  }
  MLS_STATE.running = false;
}

function waitForPairDone(timeoutMs){
  return new Promise(resolve => {
    let done = false;
    const timer = setTimeout(() => { if (!done) { cleanup(); resolve(false); } }, timeoutMs);
    function onMsg(msg){ if (msg?.type === 'mls_done') { done = true; cleanup(); resolve(true); } }
    function cleanup(){ chrome.runtime.onMessage.removeListener(onMsg); clearTimeout(timer); }
    chrome.runtime.onMessage.addListener(onMsg);
  });
}

function stopJob(){
  MLS_STATE.running = false;
  MLS_STATE.queue = [];
}

function pauseJob(){
  // Do not clear the queue; simply stop the loop
  MLS_STATE.running = false;
}

async function resumeJob(){
  if (!MLS_STATE.queue || MLS_STATE.queue.length === 0) return;
  if (MLS_STATE.running) return;
  if (!(await isAuthorized())) return;
  MLS_STATE.running = true;
  const tabId = await ensureMapsTab();
  runQueue(tabId);
}
